#!/usr/bin/env python

# Copyright 2017 Juniper Networks, Inc. All rights reserved.
# Licensed under the Juniper Networks Script Software License (the "License").
# You may not use this script file except in compliance with the License, which is located at
# http://www.juniper.net/support/legal/scriptlicense/
# Unless required by applicable law or otherwise agreed to in writing by the parties, software
# distributed under the License is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
#
# Author: gonzalo@juniper.net

# stdlib
import sys
import requests
import json
import argparse
import random
import time
from datetime import datetime
from datetime import date, timedelta
from time import sleep 
from pprint import pprint 
from StringIO import StringIO
import os

# Needed for proper search URL construction in RES API calls
import urllib2

# Import old jsonpath library due to open license scheme
# sudo pip install http://www.ultimate.com/phil/python/download/jsonpath-0.54.tar.gz
import jsonpath

# Robot libraries

from robot.libraries.BuiltIn import BuiltIn
from robot.libraries.OperatingSystem import OperatingSystem
from robot.api import logger

# Workaround for REST API over SSL with python < 2.7.9
requests.packages.urllib3.disable_warnings()

# Global Variables
timestamp =  datetime.now().strftime("%Y-%m-%d")
timestamp2 =  datetime.now().strftime("%Y-%m-%d-%H-%M-%S.%f")[:-3]
timestamp3 = datetime.now().strftime ("%H_%M_%S")
timestamp4 = datetime.now().strftime ("%Y_%m_%d_%H_%M_%S")

class ContinuableError(RuntimeError):
    ROBOT_CONTINUE_ON_FAILURE = True

class FatalError(RuntimeError):
    ROBOT_EXIT_ON_FAILURE = True

class pybot_jnorthstar (object):

    ROBOT_LIBRARY_SCOPE = 'TEST SUITE'
    ROBOT_LISTENER_API_VERSION = 2
    
    # -----------------------------------------------------------------------
    # CONSTRUCTOR
    # -----------------------------------------------------------------------
    def __init__(self,**kvargs):

            # Setting credentials 
            
            self.grant_type = kvargs['grant_type']
            self.username = kvargs['username']
            self.password = kvargs['password']
            self.ROBOT_LIBRARY_LISTENER = self
            self.api_server = kvargs['api_server']
            self.api_port = kvargs['api_port']
            self.igp = kvargs['igp']
            self.base_url = 'https://{0}:{1}'.format(self.api_server, self.api_port)


            # Get specific Northstar REST API variables 
            variables = BuiltIn().get_variables()

            # URLs constructed using global northstar REST API variables
            self.auth_path = variables['${northstar_auth_path}']
            self.topology_path = variables['${northstar_topology_path}']
            self.auth_url = self.base_url + variables['${northstar_auth_path}']
            self.get_topologies_url = self.base_url + variables['${northstar_topology_path}']
            self.search_node_path = variables['${northstar_search_node_path}']
            self.get_nodes_path = variables['${northstar_get_nodes_path}']
            self.search_link_path = variables['${northstar_search_link_path}']
            self.get_links_path = variables['${northstar_get_links_path}']
            self.get_lsps_path = variables['${northstar_get_lsps_path}']
            self.get_bulk_lsps_path = variables['${northstar_get_bulk_lsps_path}']
            self.search_lsp_path = variables['${northstar_search_lsp_path}']

            # Value separator to be considered for input yml file 
            self.input_variable_separator = variables['${input_yml_file_list_separator}']

            # Relevant JSON paths
            if self.igp == "ISIS":
                self.rest_response_node_json_routerId = variables['${northstar_rest_response_node_json_isis_routerId}']
            elif self.igp == "OSPF":
                self.rest_response_node_json_routerId = variables['${northstar_rest_response_node_json_ospf_routerId}']
            else: 
                logger.error('Incorrect value for IGP provided: %s' % (self.igp))
                raise FatalError("Did not provide adequate IGP value for Northstar API")

            self.rest_response_node_json_hostname = variables['${northstar_rest_response_node_json_hostname}']
            self.rest_response_node_json_name = variables['${northstar_rest_response_node_json_name}']
            self.rest_response_link_json_endA_node_name = variables['${northstar_rest_response_link_json_endA_node_name}']
            self.rest_response_link_json_endZ_node_name = variables['${northstar_rest_response_link_json_endZ_node_name}']
            self.rest_response_node_pcep_status = variables['${northstar_rest_response_node_json_pcep_status}']
            self.rest_response_lsp_json_pce_initiated = variables['${northstar_rest_response_lsp_json_pce_initiated}']

            self.token = {}
            # Internal data obtained through functions
            self.topology_index = {}
            # Handlers for specific URLs 
            self.search_node_url = {}

    # -----------------------------------------------------------------------
    # FUNCTIONS START HERE
    # -----------------------------------------------------------------------

    def return_token(self):

        """
            Function that returns the internal Northstar 2.0 token value, saved as an attribute
        
        """
        return self.token

    def get_token(self):

        """
            Function that aims at gaining Northstar authentication (up to 4 attempts) and saves token value as an attribute

            Northstar authentication is based on internal attributes instantiated at creation time, namely:
                - username
                - password
                - grant_type
                - Server details
                - Auth path

        """

        headers = {}
        headers['content-type'] = 'application/json'

        data = {}
        data['grant_type'] = self.grant_type
        data['username'] = self.username
        data['password'] = self.password

        logger.debug('Trying to gain authentication with Northstar using input data: %s and auth url: %s'% (data, self.auth_url))

        # Attempting push
        for i in range(0, 3):
            try: 
                response = requests.post(self.auth_url,auth=('[USERNAME]', '[PASSWORD]'),data=json.dumps(data),headers=headers,verify=False)
                logger.debug('Correct response from Northstar API',html=True)
                break
            except Timeout as e:
                # Maybe set up for a retry, or continue in a retry loop
                if i < 2:
                    continue
                else:
                    logger.error('Error trying to connect to Northstar API, check IP connectivity to Northstar server: %s'% e)
                    sys.exit(1)
            except Exception as e:
                logger.error('Error trying to connect to Northstar API, check Northstar API status: %s'% e)
                sys.exit(1)
                raise FatalError("Could not gain authentication with Northstar API")

        #  Return code 201 when new object is created and 200 for OK (existing)
        if (response.status_code) != 201 and (response.status_code != 200):
            logger.error('Error from Northstar API Rest call, response code: %s' % response.status_code)
            raise FatalError("Could not gain authentication with Northstar API")

        # If response is Unauthorized, bad parameters have been entered as input data
        if response.text == 'Unauthorized':
            logger.error('Authentication Error with Northstar API')
            raise FatalError("Could not gain authentication with Northstar API")

        # Attempt to decode token from response
        try:
            self.token = json.loads(response.text)['access_token']
            logger.debug('Authentication token gained from Northstar API: %s' % self.token)
        except Exception as e:
            logger.error('Error trying to connect to Northstar API, check Northstar API status: %s'% e)
            sys.exit(1)
            raise FatalError("Could not gain authentication with Northstar API")

    def get_rest_op(self,path,params=None):

        """
            Generic REST operation in the context of Northstar API

            Assumes that token has been previously obtained and takes full URL path as argument

            Params can be non if all existing details are requested from the URI

            Returns response json file or raises ContinuableError otherwise

        """

        # Authentication header based on bearer token
        if self.token != {}:
            # Token is not empty, construct Auth bearer header
            headers = {}
            headers['content-type'] = 'application/json'
            headers['Authorization'] = 'Bearer ' + self.token
            logger.debug('Authentication Bearer header for Northstar API: %s' % (headers))
        else:
            raise FatalError("Authentication token for Northstar API is not present")

        # Issue specific request based on path argument and obviating strict SSL checking
        # Attempt operation up to 4 times
        for i in range(0, 3):
            try: 
                response = requests.get(path,headers=headers,params=params,verify=False)
                logger.debug('Correct response from Northstar API GET operation for path: %s' % (path))
                break
            except Timeout as e:
                # Maybe set up for a retry, or continue in a retry loop
                if i < 2:
                    continue
                else:
                    logger.error('Error trying to connect to Northstar API, check IP connectivity to Northstar server: %s'% (e))
                    sys.exit(1)
            except Exception as e:
                logger.error('Error trying to connect to Northstar API, check Northstar API status: %s'% (e))
                sys.exit(1)
                raise ContinuableError("Could not issue GET REST operation with Northstar API")

        #  Return code 201 when new object is created and 200 for OK (existing)
        if (response.status_code) != 201 and (response.status_code != 200):
            logger.error('Error from Northstar API Rest call, response code: %s' % (response.status_code))
            raise ContinuableError("Could not issue GET REST operation with Northstar API")

        # If response is Unauthorized, bad parameters have been entered as input data
        if response.text == 'Unauthorized':
            logger.error('Authentication Error with Northstar API')
            raise ContinuableError("Could not issue GET REST operation with Northstar API")

        # If everything is correct return response in JSON format
        return json.loads(response.text)

    def post_rest_op(self,path,jsondata):

        """
            Generic POST operation in the context of Northstar API

            Assumes that token has been previously obtained and takes full URL path as argument

            Returns response json file or raises ContinuableError otherwise

        """

        # Authentication header based on bearer token
        if self.token != {}:
            # Token is not empty, construct Auth bearer header
            headers = {}
            headers['content-type'] = 'application/json'
            headers['Authorization'] = 'Bearer ' + self.token
            logger.debug('Authentication Bearer header for Northstar API: %s' % (headers))
        else:
            raise FatalError("Authentication token for Northstar API is not present")

        # Issue specific request based on path argument and obviating strict SSL checking
        # Attempt operation up to 4 times
        for i in range(0, 3):
            try: 
                response = requests.post(path,headers=headers,json=jsondata,verify=False)
                logger.debug('Correct response from Northstar API POST operation for path: %s' % (path))
                break
            except Timeout as e:
                # Maybe set up for a retry, or continue in a retry loop
                if i < 2:
                    continue
                else:
                    logger.error('Error trying to connect to Northstar API, check IP connectivity to Northstar server: %s'% (e))
                    sys.exit(1)
            except Exception as e:
                logger.error('Error trying to connect to Northstar API, check Northstar API status: %s'% (e))
                sys.exit(1)
                raise ContinuableError("Could not issue POST REST operation with Northstar API")

        #  Return code 202 when object is updated and 200 for OK (existing)
        if (response.status_code) != 201 and (response.status_code != 200):
            logger.error('Error from Northstar API Rest call, response code: %s and text %s' % (response.status_code, response.text))
            raise ContinuableError("Could not issue POST REST operation with Northstar API")

        # If response is Unauthorized, bad parameters have been entered as input data
        if response.text == 'Unauthorized':
            logger.error('Authentication Error with Northstar API')
            raise ContinuableError("Could not issue POST REST operation with Northstar API")

        # If everything is correct return response in JSON format
        return json.loads(response.text)

    def put_rest_op(self,path,jsondata):

        """
            Generic PUT operation in the context of Northstar API

            Assumes that token has been previously obtained and takes full URL path as argument

            Returns response json file or raises ContinuableError otherwise

        """

        # Authentication header based on bearer token
        if self.token != {}:
            # Token is not empty, construct Auth bearer header
            headers = {}
            headers['content-type'] = 'application/json'
            headers['Authorization'] = 'Bearer ' + self.token
            logger.debug('Authentication Bearer header for Northstar API: %s' % (headers))
        else:
            raise FatalError("Authentication token for Northstar API is not present")

        # Issue specific request based on path argument and obviating strict SSL checking
        # Attempt operation up to 4 times
        for i in range(0, 3):
            try: 
                response = requests.put(path,headers=headers,json=jsondata,verify=False)
                logger.debug('Correct response from Northstar API PUT operation for path: %s' % (path))
                break
            except Timeout as e:
                # Maybe set up for a retry, or continue in a retry loop
                if i < 2:
                    continue
                else:
                    logger.error('Error trying to connect to Northstar API, check IP connectivity to Northstar server: %s'% (e))
                    sys.exit(1)
            except Exception as e:
                logger.error('Error trying to connect to Northstar API, check Northstar API status: %s'% (e))
                sys.exit(1)
                raise ContinuablelError("Could not issue PUT REST operation with Northstar API")

        #  Return code 202 when object is updated and 200 for OK (existing)
        if (response.status_code) != 202 and (response.status_code != 200):
            logger.error('Error from Northstar API Rest call, response code: %s and text %s' % (response.status_code, response.text))
            raise ContinuableError("Could not issue PUT REST operation with Northstar API")

        # If response is Unauthorized, bad parameters have been entered as input data
        if response.text == 'Unauthorized':
            logger.error('Authentication Error with Northstar API')
            raise ContinuableError("Could not issue PUT REST operation with Northstar API")

        # If everything is correct return response in JSON format
        return json.loads(response.text)

    def del_rest_op(self,path,jsondata=None):

        """
            Generic DEL operation in the context of Northstar API

            Jsondata may be needed or not, depending on operation

            Assumes that token has been previously obtained and takes full URL path as argument

            Returns response json file or raises ContinuableError otherwise

        """

        # Authentication header based on bearer token
        if self.token != {}:
            # Token is not empty, construct Auth bearer header
            headers = {}
            headers['content-type'] = 'application/json'
            headers['Authorization'] = 'Bearer ' + self.token
            logger.debug('Authentication Bearer header for Northstar API: %s' % (headers))
        else:
            raise FatalError("Authentication token for Northstar API is not present")

        # Issue specific request based on path argument and obviating strict SSL checking
        # Attempt operation up to 4 times
        for i in range(0, 3):
            try: 
                response = requests.delete(path,headers=headers,json=jsondata,verify=False)
                logger.debug('Correct response from Northstar API DEL operation for path: %s' % (path))
                break
            except Timeout as e:
                # Maybe set up for a retry, or continue in a retry loop
                if i < 2:
                    continue
                else:
                    logger.error('Error trying to connect to Northstar API, check IP connectivity to Northstar server: %s'% (e))
                    sys.exit(1)
            except Exception as e:
                logger.error('Error trying to connect to Northstar API, check Northstar API status: %s'% (e))
                sys.exit(1)
                raise ContinuablelError("Could not issue DEL REST operation with Northstar API")

        #  Return code 204 when object is deleted and 200 for OK (existing)
        if (response.status_code) != 204 and (response.status_code != 200):
            logger.error('Error from Northstar API Rest call, response code: %s and text %s' % (response.status_code, response.text))
            raise ContinuableError("Could not issue DEL REST operation with Northstar API")

        # If response is Unauthorized, bad parameters have been entered as input data
        try:
            if response.text == 'Unauthorized':
                logger.error('Authentication Error with Northstar API')
                raise ContinuableError("Could not issue DEL REST operation with Northstar API")
        except:
            # If everything is correct return True (no explicit response)
            return True

    def get_topology_index(self):
        """
            GET operation to get active topology

            Recursively uses get_rest_op with self.get_topologies_url
        """

        try:
            # At the moment single topology is supported, therefore [0] as GET JSON file is a list
            self.topology_index = self.get_rest_op(path=self.get_topologies_url,params={})[0]['topologyIndex']
            logger.debug('Correct Topology index %s extracted via Northstar API GET operation' % (self.topology_index))
        except Exception as e:
            logger.error('Error trying to extract Topology, check Northstar API status: %s'% (e))
            raise FatalError("Could not extract Topology Northstar API")

    def get_nodes(self):
        """
            GET operation to get nodes from active topology

            Recursively uses get_rest_op with self.get_nodes_path
        """

        # Consider relative paths as they depend on the Topology index
        self.get_nodes_url = self.base_url + self.topology_path + '/' + str(self.topology_index) + self.get_nodes_path

        logger.debug('Querying Northstar REST API with Get nodes URL %s' % self.get_nodes_url)

        try:
            # At the moment single topology is supported, therefore [0] as GET JSON file is a list
            nodelist = self.get_rest_op(path=self.get_nodes_url,params={})
            logger.debug('Extracted complete Node list via Northstar REST API %s' % self.get_nodes_url)
            return nodelist
        except Exception as e:
            logger.error('Error trying to extract Node list, check Northstar API status: %s'% (e))
            raise FatalError("Could not extract Node list with Northstar API")

    def get_pcep_active_nodes(self):
        """
            Extract PCEP nabled routers by issuing a global GET operation to get nodes, then parsing PEP enabled routers

            Recursively uses get_nodes to obtain node list and perform parsing using plain Jsonpath
        """

        # Attempt to obtain link list via GET nodes operation
        try:
            # At the moment single topology is supported, therefore [0] as GET JSON file is a list
            nodelist = self.get_nodes()
        except Exception as e:
            logger.error('Error trying to extract Node list for attribute parsing, check Northstar API status: %s'% (e))
            raise FatalError("Could not extract Node list with Northstar API")

        # Regexp to extract PCEP enabled routers
        expr_pcep_status = '$.[?(@%s=="Up")]' % (self.rest_response_node_pcep_status)
        logger.debug('Using jsonpath expr %s to extract node PCEP status' % (expr_pcep_status))

        # Obtain Jsonpath response using regexps

        # Proceed with node extraction with PCEP enabled
        pcep_nodes = jsonpath.jsonpath(nodelist,expr_pcep_status)
        logger.debug('Node list with PCEP operational Status Up: %s' % (pcep_nodes))
        logger.debug('Type: %s' % (type(pcep_nodes)))

        # Jsonpath expression for routerId extraction
        expr = '$.%s' % (self.rest_response_node_json_routerId)
        logger.debug('Using jsonpath expr %s to extract Router Ids from PCEP enabled nodes' % (expr))

        if pcep_nodes is not False:
            # Execute jsonpath parsing to extract Router Ids from PCEP Nodes
            pcep_node_routerids = jsonpath.jsonpath(pcep_nodes,expr)
        else:
            logger.error('Could not extract PCEP enabled nodes')
            raise ContinuableError("Could not detect Nodes with PCEP active")   

        if pcep_node_routerids is not False:
            # Return List of Router Ids from PCEP Nodes
            logger.debug('Router Id list for PCEP enabled nodes: %s' % (pcep_node_routerids))
            return pcep_node_routerids
        else:
            logger.error('Could not extract Router Ids from PCEP enabled nodes from the Node List using jsonpath expr %s' % (expr))
            raise ContinuableError("Could not extract Router Ids from detected PCEP-enabled Nodes")

    def get_node_using_hostname(self,hostname):
        """
            GET operation to search node based on hostname

            Recursively uses get_rest_op with self.search_node_url
        """
        # Search params dictionary needs to be constructed
        params = {}
        # It is necessary to include trailing '$' for exact hostname matches
        params['hostname']=hostname + urllib2.unquote('$')

        # Consider relative paths as they depend on the Topology index
        self.search_node_url = self.base_url + self.topology_path + '/' + str(self.topology_index) + self.search_node_path

        logger.debug('Querying Northstar REST API with Search node URL %s'% self.search_node_url)

        try:
            # At the moment single topology is supported, therefore [0] as GET JSON file is a list
            nodelist = self.get_rest_op(path=self.search_node_url,params=params)
        except Exception as e:
            logger.error('Error trying to extract Node based on hostname, check Northstar API status: %s'% (e))
            raise FatalError("Could not extract Node Northstar API")

        if len(nodelist) == 1:
            node = nodelist[0]
            logger.debug('Extracted node details %s' % node)
            logger.debug('Correct Node index %s extracted via Northstar API GET operation' % (node['nodeIndex']))
            return node
        else:
            logger.debug('Extracted node list details %s' % nodelist)
            logger.error('Extracted more than a single node with this hostname, check hostname definition')
            raise ContinuableError("Extracted more than a single node")          

    def get_node_using_router_id(self,router_id):
        """
            GET operation to get node based on router_id

            Recursively uses get_nodes to obtain node list and perform parsing using plain Jsonpath
        """
        # Attempt to obtain node list via GET nodes operation
        try:
            # At the moment single topology is supported, therefore [0] as GET JSON file is a list
            nodelist = self.get_nodes()
        except Exception as e:
            logger.error('Error trying to extract Node list for attribute parsing, check Northstar API status: %s'% (e))
            raise FatalError("Could not extract Node list with Northstar API")

        # Regexp to extract Hostname using routerId
        expr = '$.[?(@%s=="%s")]' % (self.rest_response_node_json_routerId,router_id)
        logger.debug('Using jsonpath expr %s to extract node using routerId' % (expr))

        # Obtain Jsonpath response using regexp
        nodes = jsonpath.jsonpath(nodelist,expr)

        if len(nodes) == 1:
            node = nodes[0]
            logger.debug('Extracted node details %s' % node)
            logger.debug('Correct Node index %s extracted via Northstar API GET operation' % (node['nodeIndex']))
            return node
        else:
            logger.error('Extracted more than a single node with this hostname, check hostname definition')
            raise ContinuableError("Extracted more than a single node")  

    def get_node_hostname_using_router_id(self,router_id):
        """
            GET operation to get node based on router_id

            Recursively uses get_nodes to obtain node list and perform parsing using plain Jsonpath
        """
        # Attempt to obtain node list via GET nodes operation
        try:
            # At the moment single topology is supported, therefore [0] as GET JSON file is a list
            node = self.get_node_using_router_id(router_id)
        except Exception as e:
            logger.error('Error trying to extract Node list for hostname parsing, check Northstar API status: %s'% (e))
            raise FatalError("Could not extract Node list with Northstar API")

        # Regexp to extract Hostname using routerId
        expr = '$.%s' % (self.rest_response_node_json_hostname)
        logger.debug('Using jsonpath expr %s to extract hostname using routerId' % (expr))

        # Obtain Jsonpath response using regexp
        try:
            hostname = jsonpath.jsonpath(node,expr)
            logger.debug('Extracted hostname %s details' % hostname)
            return hostname
        except:
            logger.error('Could not extract hostname, check hostname definition')
            raise ContinuableError("Could not extract hostname")          

    def get_node_name_using_hostname(self,hostname):
        """
            GET operation to get node based on hostname

            Recursively uses get_node to search for matching node and simply extract hostname.

            This operation is normally used to obtain endpoint names for links
        """
        # Attempt to obtain node via GET node search operation
        try:
            # At the moment single topology is supported, therefore [0] as GET JSON file is a list
            node = self.get_node_using_hostname(hostname)
            logger.debug('Node %s extracted using hostname %s' % (node,hostname))
        except Exception as e:
            logger.error('Error trying to extract Node for hostname parsing, check Northstar API status: %s'% (e))
            raise FatalError("Could not extract Node list with Northstar API")

        # Regexp to extract Name using hostname
        #expr = '$.%s' % (self.rest_response_node_json_name)
        #print "*DEBUG* Using jsonpath expr %s to extract Name using hostname" % (expr)

        # Obtain Jsonpath response using regexp
        try:
            #name = jsonpath.jsonpath(node,expr)
            name = node['name']
            logger.debug('Extracted Name %s details' % name)
            return str(name)
        except:
            logger.error('Could not extract name, check name definition')
            raise ContinuableError("Could not extract name")   

    def get_links(self):
        """
            GET operation to get links from active topology

            Recursively uses get_rest_op with self.get_links_url
        """

        # Consider relative paths as they depend on the Topology index
        self.get_links_url = self.base_url + self.topology_path + '/' + str(self.topology_index) + self.get_links_path

        logger.debug('Querying Northstar REST API with Get links URL %s' % self.get_links_url)

        try:
            # At the moment single topology is supported, therefore [0] as GET JSON file is a list
            linklist = self.get_rest_op(path=self.get_links_url,params={})
            logger.debug('Extracted complete Link list via Northstar REST API %s' % self.get_links_url)
            return linklist
        except Exception as e:
            logger.error('Error trying to extract Link list, check Northstar API status: %s'% (e))
            raise FatalError("Could not extract Link list with Northstar API")

    def get_links_using_node_names(self,node1,node2):
        """
            GET operation to get link based on names from both endpoints

            Recursively uses get_links to obtain node list and perform parsing using plain Jsonpath
        """

        # Attempt to obtain link list via GET nodes operation
        try:
            # At the moment single topology is supported, therefore [0] as GET JSON file is a list
            linklist = self.get_links()
        except Exception as e:
            logger.error('Error trying to extract Link list for attribute parsing, check Northstar API status: %s'% (e))
            raise FatalError("Could not extract Link list with Northstar API")

        # Regexps to extract Link using Node names either as endA or endZ
        expr_node1_endA = '$.[?(@%s=="%s")]' % (self.rest_response_link_json_endA_node_name,node1)
        logger.debug('Using jsonpath expr %s to extract node using node name %s' % (expr_node1_endA,node1))

        expr_node1_endZ = '$.[?(@%s=="%s")]' % (self.rest_response_link_json_endZ_node_name,node1)
        logger.debug('Using jsonpath expr %s to extract node using node name %s' % (expr_node1_endZ,node1))

        expr_node2_endA = '$.[?(@%s=="%s")]' % (self.rest_response_link_json_endA_node_name,node2)
        logger.debug('Using jsonpath expr %s to extract node using node name %s' % (expr_node2_endA,node2))

        expr_node2_endZ = '$.[?(@%s=="%s")]' % (self.rest_response_link_json_endZ_node_name,node2)
        logger.debug('Using jsonpath expr %s to extract node using node name %s' % (expr_node2_endZ,node2))


        # Obtain Jsonpath response using regexps

        # Proceed with node1 first as endA
        links_node1_endA = jsonpath.jsonpath(linklist,expr_node1_endA)
        logger.debug('Link list with first Node as endA: %s' % (links_node1_endA))

        if links_node1_endA is not False:
            # Execute jsonpath parsing for the link subset where Node 1 is End A
            links_node1_endA_node2_endZ = jsonpath.jsonpath(links_node1_endA,expr_node2_endZ)
            logger.debug('Link list with first Node as endA and second node as endZ: %s' % (links_node1_endA_node2_endZ))
        else:
            links_node1_endA_node2_endZ = False
            logger.debug('No links found with first Node as endA and second node as endZ') 


        # Proceed with node1 first as endZ
        links_node1_endZ = jsonpath.jsonpath(linklist,expr_node1_endZ)
        logger.debug('Link list with first Node as endZ: %s' % (links_node1_endZ))

        if links_node1_endZ is not False:
            # Execute jsonpath parsing for the link subset where Node 1 is End Z
            links_node1_endZ_node2_endA = jsonpath.jsonpath(links_node1_endZ,expr_node2_endA)
            logger.debug('Link list with first Node as endZ and second node as endA: %s' % (links_node1_endZ_node2_endA))
        else:
            links_node1_endZ_node2_endA = False
            logger.debug('No links found with first Node as endZ and second node as endA') 

        # Final node extraction or concatenantion
        if links_node1_endA_node2_endZ is False:
            if links_node1_endZ_node2_endA is False:
                logger.error('No links found with either Node as endA or endZ')
                raise ContinuableError("No links could be found with either Node as enda or endZ")
            else:
                logger.debug('Found link list: %s' % (links_node1_endZ_node2_endA))
                return links_node1_endZ_node2_endA
        # Found links with first node as endA and second node as endZ
        else:
            if links_node1_endZ_node2_endA is False:
                logger.debug('Found link list: %s' % (links_node1_endA_node2_endZ))
                return links_node1_endA_node2_endZ
            else:
                links_node1_node2 = links_node1_endA_node2_endZ + links_node1_endZ_node2_endA
                logger.debug('Concatenating both dictionaries as nodes are found in both endpoints: %s' % (links_node1_node2))
                return links_node1_node2 

    def get_lsp_using_name_and_source(self,source,lspname):
        """
            GET operation to search LSP based on source and hostname

            Recursively uses get_rest_op with self.search_lsp_url
        """
        # Search params dictionary needs to be constructed
        params = {}
        # It is necessary to include trailing '$' for exact matches
        params['name']=lspname + urllib2.unquote('$')
        params['from']=source + urllib2.unquote('$')

        # Consider relative paths as they depend on the Topology index
        self.search_lsp_url = self.base_url + self.topology_path + '/' + str(self.topology_index) + self.search_lsp_path

        logger.debug('Querying Northstar REST API with Search LSP URL %s' % self.search_lsp_url)

        try:
            # At the moment single topology is supported, therefore [0] as GET JSON file is a list
            lsplist = self.get_rest_op(path=self.search_lsp_url,params=params)
        except Exception as e:
            logger.error('Error trying to extract LSP based on name and source, check Northstar API status: %s'% (e))
            raise FatalError("Could not extract LSP via Northstar API")

        if len(lsplist) == 1:
            lsp = lsplist[0]
            logger.debug('Extracted LSP details %s' % lsp)
            logger.debug('Correct LSP index %s extracted via Northstar API GET operation' % (lsp['lspIndex']))
            return lsp
        else:
            logger.error('Extracted more than a single LSP with this name and source, check name definition')
            raise ContinuableError("Extracted more than a LSP")

    def get_pce_initiated_lsps(self):
        """
            GET operation to obtain and provide all PCE initiated LSPs

            Recursively uses get_rest_op with self.search_lsp_path
        """

        # Consider relative paths as they depend on the Topology index
        self.search_lsp_url = self.base_url + self.topology_path + '/' + str(self.topology_index) + self.search_lsp_path

        logger.debug('Querying Northstar REST API with Search LSP URL %s' % self.search_lsp_url)

        try:
            # At the moment single topology is supported, therefore [0] as GET JSON file is a list
            lsplist = self.get_rest_op(path=self.search_lsp_url)
        except Exception as e:
            logger.error('Error trying to extract LSP list, check Northstar API status: %s'% (e))
            raise FatalError("Could not extract LSP via Northstar API")

        # Using Jsonpath expression to obtain PCE initiated LSPs 
        expr = '$.[?(@%s=="PCEInitiated")]' % (self.rest_response_lsp_json_pce_initiated)
        logger.debug('Using jsonpath expr %s to extract node using routerId' % (expr))

        # Obtain Jsonpath response using regexp
        lsps = jsonpath.jsonpath(lsplist,expr)

        if lsps is not False:
            logger.debug('Extracted PCE initiated LSP details %s' % lsps)
            return lsps
        else:
            logger.error('Could not extract any PCE initiated LSP')
            raise ContinuableError("Could not extract any PCE initiated LSP")

    def update_link_attributes(self,link,**kvargs):
        """
            PUT operation to update link attributes based on names from both endpoints

            Recursively uses put_rest_op to update existing link object
        """

        logger.debug('Link to update: %s' % (link))
        # A single link is passed over, TEcolorA and TEcolorB need to be updated
        try:
            # Updated link is first link from list
            updated_link = link
            updated_link.pop('name',None)
            updated_link.pop('operationalStatus',None)
            # Attrbiute extraction from function
            if 'TEcolorA' in kvargs.keys():
                try:
                    updated_link['endA']['TEcolor'] = int(kvargs['TEcolorA'])
                except:
                    updated_link['endA'].pop('TEcolor',None)
            if 'TEcolorZ' in kvargs.keys():
                try:
                    updated_link['endZ']['TEcolor'] = int(kvargs['TEcolorZ'])
                except:
                    updated_link['endZ'].pop('TEcolor',None)
            if 'TEmetricA' in kvargs.keys():
                try:
                    updated_link['endA']['TEmetric'] = int(kvargs['TEmetricA'])
                except:
                    updated_link['endA'].pop('TEmetric',None)
            if 'TEmetricZ' in kvargs.keys():
                try:
                    updated_link['endZ']['TEmetric'] = int(kvargs['TEmetricZ'])
                except:
                    updated_link['endZ'].pop('TEmetric',None)
            if 'delayA' in kvargs.keys():
                try:
                    updated_link['endA']['delay'] = int(kvargs['delayA'])
                except:
                    updated_link['endA'].pop('delay',None)
            if 'delayZ' in kvargs.keys():
                try:
                    updated_link['endZ']['delay'] = int(kvargs['delayZ'])
                except:
                    updated_link['endZ'].pop('delay',None)
            if 'srlgA' in kvargs.keys():
                # Assume it is always a string
                # Use variable value separate to split string into possible values
                # and create a list of srlgs out of it, even though it could have a single member only
                srlg_list = str(kvargs['srlgA']).split(self.input_variable_separator)

                # Reset existing value and iterate through list
                updated_link['endA']['srlgs'] = []
                try:
                    for i in range(0, len(srlg_list)):
                        updated_link['endA']['srlgs'].append({"srlgValue": int(srlg_list[i])})
                except:
                    pass

                # Apply similar procedure depending on IGP
                if self.igp == "ISIS":
                    updated_link['endA']['protocols']['ISIS']['srlgs'] = []
                    try:
                        for i in range(0, len(srlg_list)):
                            updated_link['endA']['protocols']['ISIS']['srlgs'].append({"srlgValue": int(srlg_list[i])})
                    except:
                        pass

                if self.igp == "OSPF":
                    updated_link['endA']['protocols']['OSPF']['srlgs'] = []
                    try:
                        for i in range(0, len(srlg_list)):
                            updated_link['endA']['protocols']['OSPF']['srlgs'].append({"srlgValue": int(srlg_list[i])})
                    except:
                        pass
            if 'srlgZ' in kvargs.keys():
                # Assume it is always a string
                # Use variable value separate to split string into possible values
                # and create a list of srlgs out of it, even though it could have a single member only
                srlg_list = str(kvargs['srlgZ']).split(self.input_variable_separator)

                # Reset existing value and iterate through list
                updated_link['endZ']['srlgs'] = []
                try:
                    for i in range(0, len(srlg_list)):
                        updated_link['endZ']['srlgs'].append({"srlgValue": int(srlg_list[i])})
                except:
                    pass

                # Apply similar procedure depending on IGP
                if self.igp == "ISIS":
                    updated_link['endZ']['protocols']['ISIS']['srlgs'] = []
                    try:
                        for i in range(0, len(srlg_list)):
                            updated_link['endZ']['protocols']['ISIS']['srlgs'].append({"srlgValue": int(srlg_list[i])})
                    except:
                        pass
                if self.igp == "OSPF":
                    updated_link['endZ']['protocols']['OSPF']['srlgs'] = []
                    try:
                        for i in range(0, len(srlg_list)):
                            updated_link['endZ']['protocols']['OSPF']['srlgs'].append({"srlgValue": int(srlg_list[i])})
                    except:
                        pass
            #updated_link['endA'].pop('unreservedBw',None)
            #updated_link['endZ'].pop('unreservedBw',None)
        except Exception as e:
            logger.error('Error %s trying to update type %s Link' % (e,updated_link))
            raise FatalError("Could not update Link attributes")

        logger.debug('Updated Link: %s' % (updated_link))

        # Consider relative paths as they depend on the Topology index
        self.put_link_url = self.base_url + self.topology_path + '/' + str(self.topology_index) + self.get_links_path + '/' + str(updated_link['linkIndex'])

        logger.debug('Putting new link config through Northstar REST API with specific link URL %s' % self.put_link_url)

        try:
            # At the moment single topology is supported, therefore [0] as GET JSON file is a list
            response = self.put_rest_op(path=self.put_link_url,jsondata=updated_link)
        except Exception as e:
            logger.error('Error trying to update Link, check Northstar API status: %s'% (e))
            raise FatalError("Could not update Link with Northstar API")

    def update_node_attributes(self,node,**kvargs):
        """
            PUT operation to update Node attributes 

            Recursively uses put_rest_op to update existing node object
        """

        logger.debug('Node to update: %s' % (node))
        # A single node is passed over
        try:
            # Updated link is first link from list
            updated_node = node
            #updated_node.pop('name',None)
            updated_node.pop('operationalStatus',None)
            updated_node.pop('protocols',None)
            updated_node['protocols'] = {}
            # Attrbiute extraction from function
            if 'hostname' in kvargs.keys():
                updated_node['hostName'] = kvargs['hostname']
            if 'nodeType' in kvargs.keys():
                updated_node['nodeType'] = kvargs['nodeType']
            if 'site' in kvargs.keys():
                updated_node['site'] = kvargs['site']
            if 'coordinates' in kvargs.keys():
                updated_node['topology'] = {}
                updated_node['topology']['coordinates'] = {}
                updated_node['topology']['coordinates']['type'] = "Point"
                updated_node['topology']['coordinates']['coordinates'] = kvargs['coordinates']

        except Exception as e:
            logger.error('Error %s trying to update type %s Node' % (e,updated_node))
            raise FatalError("Could not update Node attributes")

        logger.debug('Updated Node: %s' % (updated_node))

        # Consider relative paths as they depend on the Topology index
        self.put_node_url = self.base_url + self.topology_path + '/' + str(self.topology_index) + self.get_nodes_path + '/' + str(updated_node['nodeIndex'])

        logger.debug('Putting new node config through Northstar REST API with specific node URL %s' % self.put_node_url)

        try:
            # At the moment single topology is supported, therefore [0] as GET JSON file is a list
            response = self.put_rest_op(path=self.put_node_url,jsondata=updated_node)
        except Exception as e:
            logger.error('Error trying to update Node, check Northstar API status: %s'% (e))
            raise FatalError("Could not update Node with Northstar API")

    def construct_lsp_attributes (self,lsp_name,**kvargs):
        """
            Auxiliary function to construct LSP attributes as a dictionary for POST TE-LSP and TE-LSP bulk operationalStatus

            Takes variables as input data
        """
        try:
            lsp = {}
            lsp['name'] = lsp_name
            logger.debug('LSP name: %s' % (lsp_name))
            # Compulsory attribute extraction from function
            if 'source' in kvargs.keys():
                lsp['from'] = {}
                lsp['from']['topoObjectType'] = "ipv4"
                lsp['from']['address'] = kvargs['source']
                logger.debug('LSP source: %s' % (kvargs['source']))
            if 'destination' in kvargs.keys():
                lsp['to'] = {}
                lsp['to']['topoObjectType'] = "ipv4"
                lsp['to']['address'] = kvargs['destination']
                logger.debug('LSP destination: %s' % (kvargs['destination']))
        except Exception as e:
            logger.error('Error %s trying to create LSP' % (e))
            raise ContinuableError("Could not find any of the compulsory LSP attributes for creation: name, source or destination")

        # Planned Properties for Hold and Setup (assume 7 if not present)
        lsp['plannedProperties'] = {}

        if 'setupPriority' in kvargs.keys():
            lsp['plannedProperties']['setupPriority'] = int(kvargs['setupPriority'])
        else:
            # Default minimum prio
            lsp['plannedProperties']['setupPriority'] = 7
        logger.debug('LSP to create with setupPriority %s' % kvargs['setupPriority'])
        

        if 'holdingPriority' in kvargs.keys():
            lsp['plannedProperties']['holdingPriority'] = int(kvargs['holdingPriority'])
        else:
            # Default minimum prio
            lsp['plannedProperties']['holdingPriority'] = 7
        logger.debug('LSP to create with holdingPriority %s' % kvargs['holdingPriority'])

        # Planned TE metric
        if 'metric' in kvargs.keys():
            lsp['plannedProperties']['metric'] = int(kvargs['metric'])
            logger.debug('LSP to create with metric %s"'% lsp['plannedProperties']['metric'])

        # Extract pathtype optional attribute
        if 'pathType' in kvargs.keys():
            if kvargs['pathType'] not in ['primary','secondary']:
                logger.error('LSP pathType attribute is not primary or secondary, but %s'% kvargs['pathType'])
                raise ContinuableError("Optional LSP attribute pathType with incorrect syntax") 
            else:
                lsp['pathType'] = kvargs['pathType']
                logger.debug('LSP to create with pathType %s' % kvargs['pathType'])

        # Optional planned Properties for BW and specific pathName
        if 'bandwidth' in kvargs.keys():
            lsp['plannedProperties']['bandwidth'] = kvargs['bandwidth']
            logger.debug('LSP to create with bandwidth %s'% kvargs['bandwidth'])
        if 'pathName' in kvargs.keys():
            lsp['plannedProperties']['pathName'] = kvargs['pathName']
            logger.debug('LSP to create with pathName %s' % kvargs['pathName'])

        # Additional optional design attributes 
        design = {}

        # Applying routing methodology, if any 
        if 'routingMethod' in kvargs.keys():
            # Default assignment
            design['routingMethod'] = "default"
            if kvargs['routingMethod'] in ['adminWeight','delay','constant','distance']:
                design['routingMethod'] = kvargs['routingMethod']
        # Applying max Hops, if any
        if 'maxHop' in kvargs.keys():
            design['maxHop'] = int(kvargs['maxHop'])
        # Applying max Delay, if any
        if 'maxDelay' in kvargs.keys():
            design['maxDelay'] = int(kvargs['maxDelay'])
        # Applying max Delay, if any
        if 'maxCost' in kvargs.keys():
            design['maxCost'] = int(kvargs['maxCost'])
        # Applying routing diversity Level, if any 
        if 'diversityLevel' in kvargs.keys():
            # Default assignment
            design['diversityLevel'] = "site"
            if kvargs['diversityLevel'] in ['srlg','link']:
                design['diversityLevel'] = kvargs['diversityLevel']
        # Applying diversityGroup as an arbitrary string
        if 'diversityGroup' in kvargs.keys():
            design['diversityGroup'] = kvargs['diversityGroup']
        # Applying useProtectedLinks as an arbitrary knob, assume "Protected" if it exists
        if 'useProtectedLinks' in kvargs.keys():
            design['useProtectedLinks'] = "Protected"

        # Final check if any design attributes have been provided
        if design:
            lsp['plannedProperties']['design'] = design

        # Return LSP attributes
        logger.debug('LSP attributes created: %s' % lsp)
        return lsp

    def update_lsp_attributes (self,oldlsp,**kvargs):
        """
            Auxiliary function to update LSP attributes as a dictionary for PUT TE-LSP and TE-LSP bulk operationalStatus

            Takes variables as input data
        """
        # Copy previous LSP values
        lsp = oldlsp

        # Drop live properties, operationalStatus, controlType and tunnelId
        lsp.pop('liveProperties',None)
        lsp.pop('operationalStatus',None)
        lsp.pop('controlType',None)
        lsp.pop('tunnelId',None)

        try:
            lsp['name'] = oldlsp['name']
            logger.debug('LSP name: %s' % (lsp['name']))
            # Compulsory attribute extraction from function
            if 'source' in kvargs.keys():
                lsp['from'] = {}
                lsp['from']['topoObjectType'] = "ipv4"
                lsp['from']['address'] = kvargs['source']
                logger.debug('LSP source: %s' % (kvargs['source']))
            if 'destination' in kvargs.keys():
                lsp['to'] = {}
                lsp['to']['topoObjectType'] = "ipv4"
                lsp['to']['address'] = kvargs['destination']
                logger.debug('LSP destination: %s' % (kvargs['destination']))
        except Exception as e:
            logger.error('Error %s trying to create LSP' % (e))
            raise ContinuableError("Could not find any of the compulsory LSP attributes for creation: name, source or destination")

        # Planned Properties for Hold and Setup 

        if 'setupPriority' in kvargs.keys():
            lsp['plannedProperties']['setupPriority'] = int(kvargs['setupPriority'])
        logger.debug('LSP to update with setupPriority %s' % lsp['plannedProperties']['setupPriority'])
        
        if 'holdingPriority' in kvargs.keys():
            lsp['plannedProperties']['holdingPriority'] = int(kvargs['holdingPriority'])
        logger.debug('LSP to create with holdingPriority %s' % lsp['plannedProperties']['holdingPriority'])

        # New updated metric as planned property

        if 'metric' in kvargs.keys():
            lsp['plannedProperties']['metric'] = int(kvargs['metric'])
            logger.debug('LSP to update with metric %s' % lsp['plannedProperties']['metric'])

        # Extract pathtype optional attribute
        if 'pathType' in kvargs.keys():
            if kvargs['pathType'] not in ['primary','secondary']:
                logger.error('LSP pathType attribute is not primary or secondary, but %s'% kvargs['pathType'])
                raise ContinuableError("Optional LSP attribute pathType with incorrect syntax") 
            else:
                lsp['pathType'] = kvargs['pathType']
                logger.debug('LSP with pathType %s' % kvargs['pathType'])

        # Optional planned Properties for BW and specific pathName
        if 'bandwidth' in kvargs.keys():
            lsp['plannedProperties']['bandwidth'] = kvargs['bandwidth']
            logger.debug('LSP with bandwidth %s' % kvargs['bandwidth'])
        if 'pathName' in kvargs.keys():
            lsp['plannedProperties']['pathName'] = kvargs['pathName']
            logger.debug('LSP with pathName %s' % kvargs['pathName'])


        # Additional optional design attributes 
        try:
            design = oldlsp['plannedProperties']['design']
        except:
            design = {}

        # Applying routing methodology, if any 
        if 'routingMethod' in kvargs.keys():
            # Default assignment
            design['routingMethod'] = "default"
            if kvargs['routingMethod'] in ['adminWeight','delay','constant','distance']:
                design['routingMethod'] = kvargs['routingMethod']

        # Applying max Hops, if any
        if 'maxHop' in kvargs.keys():
            design['maxHop'] = int(kvargs['maxHop'])
        # Applying max Delay, if any
        if 'maxDelay' in kvargs.keys():
            design['maxDelay'] = int(kvargs['maxDelay'])
        # Applying max Delay, if any
        if 'maxCost' in kvargs.keys():
            design['maxCost'] = int(kvargs['maxCost'])
        # Applying routing diversity Level, if any 
        if 'diversityLevel' in kvargs.keys():
            # Default assignment
            design['diversityLevel'] = "site"
            if kvargs['diversityLevel'] in ['srlg','link']:
                design['diversityLevel'] = kvargs['diversityLevel']
        # Applying diversityGroup as an arbitrary string
        if 'diversityGroup' in kvargs.keys():
            design['diversityGroup'] = kvargs['diversityGroup']
        # Applying useProtectedLinks as an arbitrary knob, assume "Protected" if it exists
        if 'useProtectedLinks' in kvargs.keys():
            design['useProtectedLinks'] = "Protected"

        # Final check if any design attributes have been provided
        if design:
            lsp['plannedProperties']['design'] = design

        # Return LSP attributes
        logger.debug('Updated LSP attributes: %s' % lsp)
        return lsp

    def create_node(self,hostname):
        """
            POST operation to create planned node based on hostname (compulsory) 

            Recursively uses post_rest_op to create new planned node object
        """
        try: 
            node = {}
            node['name'] = hostname
            node['hostName'] = hostname
            node['topoObjectType'] = "node"
            node['topologyIndex'] = int(self.topology_index)
            # Dumping new node to create
            logger.debug('New Planned node to be created: %s' % node)
        except Exception as e:
            logger.error('Error trying to construct Node attributes: %s'% (e))
            raise ContinuableError("Could not create Node attributes")

        # Consider relative paths as they depend on the Topology index
        self.post_node_url = self.base_url + self.topology_path + '/' + str(self.topology_index) + self.get_nodes_path

        logger.debug('Posting new Planned node config through Northstar REST API with specific node URL %s' % self.post_node_url)

        try:
            # At the moment single topology is supported, therefore [0] as GET JSON file is a list
            response = self.post_rest_op(path=self.post_node_url,jsondata=node)
        except Exception as e:
            logger.error('Error trying to create Node, check Northstar API status: %s'% (e))
            raise FatalError("Could not create Node with Northstar API")

    def create_lsp(self,lsp_name,**kvargs):
        """
            POST operation to create lsp based on minimum attributes: lsp name and IPv4 addresses for source and destination 

            Recursively uses post_rest_op to create new LSP object
        """
        try: 
            lsp = self.construct_lsp_attributes(lsp_name,**kvargs)
            # Dumping new LSP to create
            logger.debug('New TE-LSP to be created: %s' % lsp)
        except Exception as e:
            logger.error('Error trying to construct TE-LSP attributes: %s'% (e))
            raise ContinuableError("Could not create TE-LSP attributes")

        # Consider relative paths as they depend on the Topology index
        self.post_lsp_url = self.base_url + self.topology_path + '/' + str(self.topology_index) + self.get_lsps_path

        logger.debug('Posting new TE-LSP config through Northstar REST API with specific node URL %s' % self.post_lsp_url)

        try:
            # At the moment single topology is supported, therefore [0] as GET JSON file is a list
            response = self.post_rest_op(path=self.post_lsp_url,jsondata=lsp)
        except Exception as e:
            logger.error('Error trying to create TE-LSP, check Northstar API status: %s'% (e))
            raise FatalError("Could not create TE-LSP with Northstar API")

    def update_lsp(self,lsp,**kvargs):
        """
            PUT operation to update lsp based with no minimum attributes (down to empty update for recalculation) 

            Recursively uses put_rest_op to update LSP object
        """
        # LSP to be updated
        logger.debug('TE-LSP to be updated: %s' % lsp)

        try: 
            updated_lsp = self.update_lsp_attributes(lsp,**kvargs)
            # Dumping updated LSP 
            logger.debug('Updated LSP: %s' % updated_lsp)
        except Exception as e:
            logger.error('Error trying to update TE-LSP attributes: %s'% (e))
            raise ContinuableError("Could not update TE-LSP attributes")

        # Consider relative paths as they depend on the Topology index
        self.put_lsp_url = self.base_url + self.topology_path + '/' + str(self.topology_index) + self.get_lsps_path + '/' + str(updated_lsp['lspIndex'])

        logger.debug('Posting Updated TE-LSP config through Northstar REST API with specific node URL %s' % self.put_lsp_url)

        try:
            # At the moment single topology is supported, therefore [0] as GET JSON file is a list
            response = self.put_rest_op(path=self.put_lsp_url,jsondata=updated_lsp)
        except Exception as e:
            logger.error('Error trying to update TE-LSP, check Northstar API status: %s'% (e))
            raise FatalError("Could not update TE-LSP with Northstar API")


    def create_lsp_bulk(self,source,destinations,**kvargs):
        """
            POST operation to create bulk of lsps based on minimum attributes: source, list of destinations and other attributes including name regexp 

            Recursively uses post_rest_op to create new LSP object
        """
        try:
            source_name = self.get_node_hostname_using_router_id(source)[0]
            logger.debug('Correctly found TE-LSP bulk source %s' % source_name)
        except Exception as e:
            logger.error('Error trying to extract source router name: %s'% (e))
            raise ContinuableError("Could not extract source router name")

        #Initialize Bulk of LSPs
        bulk_lsps =[]

        # Iterate for every destination to construct LSP data structure
        for i in range(0, len(destinations)):
            try:
                destination_name = self.get_node_hostname_using_router_id(destinations[i])[0]
                logger.debug('Correctly found TE-LSP destination %s' % destination_name)
            except Exception as e:
                logger.error(' Error trying to extract destination router name: %s'% (e))
                raise ContinuableError("Could not extract source router name")

            try:
                prefix = str(kvargs['name_regexp'].rstrip('*'))
                logger.debug('TE-LSP prefix %s' % prefix)
                lsp_name = prefix + source_name + '-' + destination_name
                logger.debug('TE-LSP name %s' % lsp_name) 
            except Exception as e:
                logger.error('Error trying to extract prefix from input name regexp: %s'% (e))
                raise ContinuableError("Could not extract LSP prefix")

            try:
                # Updating dictionary with source and destination
                attributes = kvargs
                attributes.update({'source':source,'destination':destinations[i]})
                logger.debug('Attributes to be passed over: %s' % attributes)
            except Exception as e:
                logger.error('Error trying to craft attributes for LSP creation: %s'% (e))
                raise ContinuableError("Could not craft LSP attributes")

            try:
                lsp = self.construct_lsp_attributes(lsp_name,**attributes)
                # Dumping new LSP to create
                logger.debug('New TE-LSP to be added to bulk: %s' % lsp)
                # Append to list of TE-LSPs
                bulk_lsps.append(lsp)
            except Exception as e:
                logger.error('Error trying to construct TE-LSP attributes for bulk provisioning: %s'% (e))
                raise ContinuableError("Could not create TE-LSP attributes")

        # Bulk of TE-LSPs should be ready to load
        logger.debug('Bulk of TE-LSP to be created: %s' % bulk_lsps)

        # Consider relative paths as they depend on the Topology index
        self.post_bulk_lsp_url = self.base_url + self.topology_path + '/' + str(self.topology_index) + self.get_bulk_lsps_path

        logger.debug('Posting new Bulk of TE-LSPs config through Northstar REST API with specific node URL %s' % self.post_bulk_lsp_url)

        try:
            # At the moment single topology is supported, therefore [0] as GET JSON file is a list
            response = self.post_rest_op(path=self.post_bulk_lsp_url,jsondata=bulk_lsps)
            logger.debug('Northstar REST API response to TE-LSP bulk provisioning: %s' % response)
        except Exception as e:
            logger.error('Error trying to create bulk of TE-LSPs, check Northstar API status: %s'% (e))
            raise FatalError("Could not create bulk of TE-LSPs with Northstar API")

    def delete_lsp(self,lsp_index):
        """
            DELETE operation to delete lsp based on lspIndex 

            Recursively uses delete_rest_op to delete LSP object
        """
        try: 
            # Dumping index from LSP to delete
            logger.debug('New TE-LSP to be created: %s' % int(lsp_index))
        except Exception as e:
            logger.error('Error trying to parse LSP Index: %s'% (e))
            raise ContinuableError("Could not parse LSP Index")

        # Consider relative paths as they depend on the Topology index
        self.del_lsp_url = self.base_url + self.topology_path + '/' + str(self.topology_index) + self.get_lsps_path + '/' + str(lsp_index)

        logger.debug('Deleting TE-LSP config through Northstar REST API with specific node URL %s' % self.del_lsp_url)

        try:
            # At the moment single topology is supported, therefore [0] as GET JSON file is a list
            response = self.del_rest_op(path=self.del_lsp_url)
        except Exception as e:
            logger.error('Error trying to delete TE-LSP, check Northstar API status: %s'% (e))
            raise FatalError("Could not delete TE-LSP with Northstar API")

    def delete_bulk_lsps(self,lsps):
        """
            DELETE operation to delete input lsps as a bulk operation 

            Recursively uses delete_rest_op to delete LSPs using bulk operations
        """

        # Consider relative paths as they depend on the Topology index
        self.del_bulk_lsp_url = self.base_url + self.topology_path + '/' + str(self.topology_index) + self.get_bulk_lsps_path
        logger.debug('Deleting TE-LSP bulk config through Northstar REST API with URL %s' % self.del_bulk_lsp_url)

        try:
            # At the moment single topology is supported, therefore [0] as GET JSON file is a list
            response = self.del_rest_op(path=self.del_bulk_lsp_url,jsondata=lsps)
        except Exception as e:
            logger.error('Error trying to delete bulk of TE-LSPs, check Northstar API status: %s'% (e))
            raise FatalError("Could not delete bulk of TE-LSPs with Northstar API")

    def reset_topology(self):
        """
            DELETE operation to reset Northstar topology 

            Recursively uses delete_rest_op for the global topology
        """

        # Consider relative paths as they depend on the Topology index
        self.reset_topo_url = self.base_url + self.topology_path + '/' + str(self.topology_index)

        logger.debug('Resetting topology through Northstar REST API with specific topology URL %s' % self.reset_topo_url)

        try:
            # At the moment single topology is supported, therefore [0] as GET JSON file is a list
            response = self.del_rest_op(path=self.reset_topo_url)
        except Exception as e:
            logger.error('Error trying to reset Topology, check Northstar API status: %s'% (e))
            raise FatalError("Could not reset topology with Northstar API")

    def delete_all_lsps(self):
        """
            DELETE operation to delete all Northstar initiated LSPs 

            Recursively uses delete_rest_op for the LSP bulk
        """

        # Consider relative paths as they depend on the Topology index
        self.del_all_lsps_url = self.base_url + self.topology_path + '/' + str(self.topology_index) + self.get_lsps_path

        logger.debug('Resetting all LSPs through Northstar REST API with specific topology URL %s' % self.del_all_lsps_url)

        try:
            # At the moment single topology is supported, therefore [0] as GET JSON file is a list
            response = self.del_rest_op(path=self.del_all_lsps_url)
        except Exception as e:
            logger.error('Error trying to delete all LSPs, check Northstar API status: %s'% (e))
            raise FatalError("Could not delete all LSPs with Northstar API")